/*
 * Decompiled with CFR 0.152.
 */
package org.kigalisim.engine.number;

import java.math.BigDecimal;
import java.math.MathContext;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.kigalisim.engine.number.EngineNumber;
import org.kigalisim.engine.state.StateGetter;
import org.kigalisim.util.UnitStringNormalizer;

public class UnitConverter {
    private static final boolean CONVERT_ZERO_NOOP = true;
    private static final boolean ZERO_EMPTY_VOLUME_INTENSITY = true;
    private static final MathContext MATH_CONTEXT = MathContext.DECIMAL128;
    private static final BigDecimal KG_TO_MT_FACTOR = new BigDecimal("1000");
    private static final BigDecimal MT_TO_KG_FACTOR = BigDecimal.ONE.divide(KG_TO_MT_FACTOR, MATH_CONTEXT);
    private static final BigDecimal PERCENT_FACTOR = new BigDecimal("100");
    private static final BigDecimal TCO2E_TO_KGCO2E_FACTOR = new BigDecimal("1000");
    private static final BigDecimal KGCO2E_TO_TCO2E_FACTOR = BigDecimal.ONE.divide(TCO2E_TO_KGCO2E_FACTOR, MATH_CONTEXT);
    private static final BigDecimal TO_PERCENT_FACTOR = BigDecimal.ONE.divide(PERCENT_FACTOR, MATH_CONTEXT);
    private static final Map<String, Map<String, BigDecimal>> SCALE_MAP = UnitConverter.createScaleMap();
    private static final Map<String, EngineNumber> CONVERSION_CACHE = new ConcurrentHashMap<String, EngineNumber>(256);
    private static final int MAX_CACHE_SIZE = 1000;
    private final StateGetter stateGetter;

    public UnitConverter(StateGetter stateGetter) {
        this.stateGetter = stateGetter;
    }

    private static Map<String, Map<String, BigDecimal>> createScaleMap() {
        HashMap scaleMap = new HashMap();
        HashMap<String, BigDecimal> kgScales = new HashMap<String, BigDecimal>();
        kgScales.put("mt", KG_TO_MT_FACTOR);
        scaleMap.put("kg", kgScales);
        HashMap<String, BigDecimal> mtScales = new HashMap<String, BigDecimal>();
        mtScales.put("kg", MT_TO_KG_FACTOR);
        scaleMap.put("mt", mtScales);
        HashMap<String, BigDecimal> unitScales = new HashMap<String, BigDecimal>();
        unitScales.put("units", BigDecimal.ONE);
        scaleMap.put("unit", unitScales);
        HashMap<String, BigDecimal> unitsScales = new HashMap<String, BigDecimal>();
        unitsScales.put("unit", BigDecimal.ONE);
        scaleMap.put("units", unitsScales);
        HashMap<String, BigDecimal> yearsScales = new HashMap<String, BigDecimal>();
        yearsScales.put("year", BigDecimal.ONE);
        scaleMap.put("years", yearsScales);
        HashMap<String, BigDecimal> yearScales = new HashMap<String, BigDecimal>();
        yearScales.put("years", BigDecimal.ONE);
        yearScales.put("yr", BigDecimal.ONE);
        yearScales.put("yrs", BigDecimal.ONE);
        scaleMap.put("year", yearScales);
        HashMap<String, BigDecimal> yrScales = new HashMap<String, BigDecimal>();
        yrScales.put("year", BigDecimal.ONE);
        yrScales.put("years", BigDecimal.ONE);
        yrScales.put("yrs", BigDecimal.ONE);
        scaleMap.put("yr", yrScales);
        HashMap<String, BigDecimal> yrsScales = new HashMap<String, BigDecimal>();
        yrsScales.put("year", BigDecimal.ONE);
        yrsScales.put("years", BigDecimal.ONE);
        yrsScales.put("yr", BigDecimal.ONE);
        scaleMap.put("yrs", yrsScales);
        return Collections.unmodifiableMap(scaleMap);
    }

    private static boolean isCacheable(String sourceUnits, String destUnits) {
        boolean isYearAlias;
        boolean isUnitAlias;
        boolean isEmissions;
        boolean isVolume;
        boolean startsWithVolume = "kg".equals(sourceUnits) || "mt".equals(sourceUnits);
        boolean endsWithVolume = "kg".equals(destUnits) || "mt".equals(destUnits);
        boolean bl = isVolume = startsWithVolume && endsWithVolume;
        if (isVolume) {
            return true;
        }
        boolean startsWithEmissions = "tCO2e".equals(sourceUnits) || "kgCO2e".equals(sourceUnits);
        boolean endsWithEmissions = "tCO2e".equals(destUnits) || "kgCO2e".equals(destUnits);
        boolean bl2 = isEmissions = startsWithEmissions && endsWithEmissions;
        if (isEmissions) {
            return true;
        }
        boolean startsWithUnits = "unit".equals(sourceUnits) || "units".equals(sourceUnits);
        boolean endsWithUnits = "unit".equals(destUnits) || "units".equals(destUnits);
        boolean bl3 = isUnitAlias = startsWithUnits && endsWithUnits;
        if (isUnitAlias) {
            return true;
        }
        boolean startsWithYears = UnitConverter.isYearVariant(sourceUnits);
        boolean endsWithYears = UnitConverter.isYearVariant(destUnits);
        boolean bl4 = isYearAlias = startsWithYears && endsWithYears;
        return isYearAlias;
    }

    private static boolean isYearVariant(String target) {
        return switch (target) {
            case "year", "years", "yr", "yrs" -> true;
            default -> false;
        };
    }

    private static String createCacheKey(String sourceUnits, String destUnits, BigDecimal value) {
        return sourceUnits + "|" + destUnits + "|" + value.toPlainString();
    }

    private static String normalizeUnitString(String unitString) {
        return UnitStringNormalizer.normalize(unitString);
    }

    private static String[] splitUnits(String unitString) {
        int slashIndex = unitString.indexOf(47);
        if (slashIndex == -1) {
            return new String[]{unitString, ""};
        }
        return new String[]{unitString.substring(0, slashIndex), unitString.substring(slashIndex + 1)};
    }

    private static String getNumerator(String unitString) {
        int slashIndex = unitString.indexOf(47);
        return slashIndex == -1 ? unitString : unitString.substring(0, slashIndex);
    }

    private boolean isYearsUnitStr(String unitString) {
        return switch (unitString) {
            case "year", "years", "yr", "yrs" -> true;
            default -> false;
        };
    }

    private boolean getEndsWithPerYear(String normalizedUnits) {
        return normalizedUnits.endsWith("/year") || normalizedUnits.endsWith("/yr") || normalizedUnits.endsWith("/yrs");
    }

    public EngineNumber convert(EngineNumber source, String destinationUnits) {
        if (source.getUnits().equals(destinationUnits)) {
            return source;
        }
        if (source.getValue().compareTo(BigDecimal.ZERO) == 0) {
            return new EngineNumber(BigDecimal.ZERO, destinationUnits);
        }
        return this.convertDifferentValue(source, destinationUnits);
    }

    private EngineNumber convertDifferentValue(EngineNumber source, String destinationUnits) {
        EngineNumber result;
        boolean sameDenominator;
        EngineNumber cached;
        String normalizedSourceUnits = UnitConverter.normalizeUnitString(source.getUnits());
        String normalizedDestinationUnits = UnitConverter.normalizeUnitString(destinationUnits);
        boolean cacheable = UnitConverter.isCacheable(normalizedSourceUnits, normalizedDestinationUnits);
        String cacheKey = null;
        if (cacheable && (cached = CONVERSION_CACHE.get(cacheKey = UnitConverter.createCacheKey(normalizedSourceUnits, normalizedDestinationUnits, source.getValue()))) != null) {
            return cached;
        }
        String[] sourceUnitPieces = UnitConverter.splitUnits(normalizedSourceUnits);
        boolean sourceHasDenominator = sourceUnitPieces.length > 1;
        String sourceDenominatorUnits = sourceHasDenominator ? sourceUnitPieces[1] : "";
        String[] destinationUnitPieces = UnitConverter.splitUnits(normalizedDestinationUnits);
        boolean destHasDenominator = destinationUnitPieces.length > 1;
        String destinationDenominatorUnits = destHasDenominator ? destinationUnitPieces[1] : "";
        String sourceNumeratorUnits = sourceUnitPieces[0];
        String destinationNumeratorUnits = destinationUnitPieces[0];
        boolean differentDenominator = !destinationDenominatorUnits.equals(sourceDenominatorUnits);
        boolean bl = sameDenominator = !differentDenominator;
        if (sourceHasDenominator && sameDenominator) {
            EngineNumber sourceEffective = new EngineNumber(source.getValue(), sourceNumeratorUnits);
            EngineNumber convertedNumerator = this.convertNumerator(sourceEffective, destinationNumeratorUnits);
            result = new EngineNumber(convertedNumerator.getValue(), destinationUnits);
        } else {
            BigDecimal inferredFactor;
            EngineNumber numerator = this.convertNumerator(source, destinationNumeratorUnits);
            EngineNumber denominator = this.convertDenominator(source, destinationDenominatorUnits);
            result = denominator.getValue().compareTo(BigDecimal.ZERO) == 0 ? ((inferredFactor = this.inferScale(sourceDenominatorUnits, destinationDenominatorUnits)) != null ? new EngineNumber(numerator.getValue().divide(inferredFactor, MATH_CONTEXT), destinationUnits) : new EngineNumber(BigDecimal.ZERO, destinationUnits)) : new EngineNumber(numerator.getValue().divide(denominator.getValue(), MATH_CONTEXT), destinationUnits);
        }
        if (cacheable && CONVERSION_CACHE.size() < 1000) {
            CONVERSION_CACHE.put(cacheKey, result);
        }
        return result;
    }

    private EngineNumber convertNumerator(EngineNumber input, String destinationUnits) {
        return switch (destinationUnits) {
            case "kg" -> this.toKg(input);
            case "mt" -> this.toMt(input);
            case "unit", "units" -> this.toUnits(input);
            case "tCO2e" -> this.toTonnesCo2eConsumption(input);
            case "kgCO2e" -> this.toKgCo2eConsumption(input);
            case "kwh" -> this.toEnergyConsumption(input);
            case "year", "years", "yr", "yrs" -> this.toYears(input);
            case "%" -> this.toPercent(input);
            default -> throw new IllegalArgumentException("Unsupported destination numerator units: " + destinationUnits);
        };
    }

    private EngineNumber convertDenominator(EngineNumber input, String destinationUnits) {
        return switch (destinationUnits) {
            case "kg" -> this.convert(this.stateGetter.getVolume(), "kg");
            case "mt" -> this.convert(this.stateGetter.getVolume(), "mt");
            case "unit", "units" -> this.convert(this.stateGetter.getPopulation(), destinationUnits);
            case "tCO2e" -> this.convert(this.stateGetter.getGhgConsumption(), "tCO2e");
            case "kgCO2e" -> this.convert(this.stateGetter.getGhgConsumption(), "kgCO2e");
            case "kwh" -> this.convert(this.stateGetter.getEnergyConsumption(), "kwh");
            case "year", "years", "yr", "yrs" -> this.convert(this.stateGetter.getYearsElapsed(), destinationUnits);
            case "" -> new EngineNumber(BigDecimal.ONE, "");
            default -> throw new IllegalArgumentException("Unsupported destination denominator units: " + destinationUnits);
        };
    }

    private BigDecimal inferScale(String source, String destination) {
        Map<String, BigDecimal> sourceScales = SCALE_MAP.get(source);
        if (sourceScales != null) {
            return sourceScales.get(destination);
        }
        return null;
    }

    private EngineNumber toKg(EngineNumber target) {
        String currentUnits;
        EngineNumber asVolume = this.toVolume(target);
        return switch (currentUnits = asVolume.getUnits()) {
            case "mt", "mteachyear" -> new EngineNumber(asVolume.getValue().multiply(KG_TO_MT_FACTOR), "kg");
            case "kg", "kgeachyear" -> new EngineNumber(asVolume.getValue(), "kg");
            default -> throw new IllegalArgumentException("Unexpected units " + currentUnits);
        };
    }

    private EngineNumber toMt(EngineNumber target) {
        String currentUnits;
        EngineNumber asVolume = this.toVolume(target);
        return switch (currentUnits = asVolume.getUnits()) {
            case "kg", "kgeachyear" -> new EngineNumber(asVolume.getValue().multiply(MT_TO_KG_FACTOR), "mt");
            case "mt", "mteachyear" -> new EngineNumber(asVolume.getValue(), "mt");
            default -> throw new IllegalArgumentException("Unexpected units " + currentUnits);
        };
    }

    private EngineNumber toVolume(EngineNumber target) {
        boolean alreadyKg;
        String currentUnits = (target = this.normalize(target)).getUnits();
        boolean alreadyMt = "mt".equals(currentUnits) || "mteachyear".equals(currentUnits);
        boolean bl = alreadyKg = "kg".equals(currentUnits) || "kgeachyear".equals(currentUnits);
        if (alreadyMt || alreadyKg) {
            return target;
        }
        return switch (currentUnits) {
            case "tCO2e" -> {
                BigDecimal originalValue = target.getValue();
                EngineNumber conversion = this.stateGetter.getSubstanceConsumption();
                String normalizedUnits = UnitConverter.normalizeUnitString(conversion.getUnits());
                String[] conversionUnitPieces = UnitConverter.splitUnits(normalizedUnits);
                String conversionNumeratorUnits = conversionUnitPieces[0];
                String newUnits = conversionUnitPieces[1];
                EngineNumber conversionNumerator = new EngineNumber(conversion.getValue(), conversionNumeratorUnits);
                EngineNumber normalizedConversionNumerator = this.convert(conversionNumerator, currentUnits);
                BigDecimal normalizedConversionValue = normalizedConversionNumerator.getValue();
                BigDecimal newValue = originalValue.divide(normalizedConversionValue, MATH_CONTEXT);
                yield new EngineNumber(newValue, newUnits);
            }
            case "kgCO2e" -> {
                BigDecimal kgco2eValue = target.getValue();
                BigDecimal tco2eValue = kgco2eValue.multiply(KGCO2E_TO_TCO2E_FACTOR);
                EngineNumber tco2eTarget = new EngineNumber(tco2eValue, "tCO2e");
                yield this.toVolume(tco2eTarget);
            }
            case "unit", "units", "unitseachyear" -> {
                BigDecimal originalValue = target.getValue();
                EngineNumber conversion = this.stateGetter.getAmortizedUnitVolume();
                BigDecimal conversionValue = conversion.getValue();
                String normalizedUnits = UnitConverter.normalizeUnitString(conversion.getUnits());
                String[] conversionUnitPieces = UnitConverter.splitUnits(normalizedUnits);
                String newUnits = conversionUnitPieces[0];
                BigDecimal newValue = originalValue.multiply(conversionValue);
                yield new EngineNumber(newValue, newUnits);
            }
            case "%" -> {
                BigDecimal originalValue = target.getValue();
                BigDecimal asRatio = originalValue.multiply(TO_PERCENT_FACTOR);
                EngineNumber total = this.stateGetter.getVolume();
                String newUnits = total.getUnits();
                BigDecimal newValue = total.getValue().multiply(asRatio);
                yield new EngineNumber(newValue, newUnits);
            }
            default -> throw new IllegalArgumentException("Unable to convert to volume: " + currentUnits);
        };
    }

    private EngineNumber toUnits(EngineNumber target) {
        String currentUnits;
        target = this.normalize(target);
        return switch (currentUnits = target.getUnits()) {
            case "units" -> target;
            case "unit", "unitseachyear" -> new EngineNumber(target.getValue(), "units");
            case "kg", "mt" -> {
                EngineNumber conversion = this.stateGetter.getAmortizedUnitVolume();
                BigDecimal conversionValue = conversion.getValue();
                String normalizedUnits = UnitConverter.normalizeUnitString(conversion.getUnits());
                String[] conversionUnitPieces = UnitConverter.splitUnits(normalizedUnits);
                String expectedUnits = conversionUnitPieces[0];
                EngineNumber targetConverted = this.convert(target, expectedUnits);
                BigDecimal originalValue = targetConverted.getValue();
                BigDecimal newValue = originalValue.divide(conversionValue, MATH_CONTEXT);
                yield new EngineNumber(newValue, "units");
            }
            case "tCO2e" -> {
                BigDecimal originalValue = target.getValue();
                EngineNumber conversion = this.stateGetter.getAmortizedUnitConsumption();
                String normalizedUnits = UnitConverter.normalizeUnitString(conversion.getUnits());
                String[] conversionUnitPieces = UnitConverter.splitUnits(normalizedUnits);
                String conversionNumeratorUnits = conversionUnitPieces[0];
                EngineNumber conversionNumerator = new EngineNumber(conversion.getValue(), conversionNumeratorUnits);
                EngineNumber normalizedConversionNumerator = this.convert(conversionNumerator, currentUnits);
                BigDecimal normalizedConversionValue = normalizedConversionNumerator.getValue();
                BigDecimal newValue = originalValue.divide(normalizedConversionValue, MATH_CONTEXT);
                yield new EngineNumber(newValue, "units");
            }
            case "kgCO2e" -> {
                BigDecimal kgco2eValue = target.getValue();
                BigDecimal tco2eValue = kgco2eValue.multiply(KGCO2E_TO_TCO2E_FACTOR);
                EngineNumber tco2eTarget = new EngineNumber(tco2eValue, "tCO2e");
                yield this.toUnits(tco2eTarget);
            }
            case "%", "%eachyear" -> {
                BigDecimal originalValue = target.getValue();
                BigDecimal asRatio = originalValue.multiply(TO_PERCENT_FACTOR);
                EngineNumber total = this.stateGetter.getPopulation();
                BigDecimal newValue = total.getValue().multiply(asRatio);
                yield new EngineNumber(newValue, "units");
            }
            default -> throw new IllegalArgumentException("Unable to convert to population: " + currentUnits);
        };
    }

    private EngineNumber toTonnesCo2eConsumption(EngineNumber target) {
        boolean currentInfer;
        String currentUnits = (target = this.normalize(target)).getUnits();
        boolean alreadyCorrect = "tCO2e".equals(currentUnits) || "tCO2eeachyear".equals(currentUnits);
        boolean currentVolume = "kg".equals(currentUnits) || "mt".equals(currentUnits);
        boolean currentPop = "unit".equals(currentUnits) || "units".equals(currentUnits);
        boolean bl = currentInfer = currentVolume || currentPop;
        if (alreadyCorrect) {
            return target;
        }
        if ("kgCO2e".equals(currentUnits)) {
            BigDecimal kgco2eValue = target.getValue();
            BigDecimal tco2eValue = kgco2eValue.multiply(KGCO2E_TO_TCO2E_FACTOR);
            return new EngineNumber(tco2eValue, "tCO2e");
        }
        if (currentInfer) {
            EngineNumber conversion = this.stateGetter.getSubstanceConsumption();
            String normalizedUnits = UnitConverter.normalizeUnitString(conversion.getUnits());
            String[] conversionUnitPieces = UnitConverter.splitUnits(normalizedUnits);
            String newUnits = conversionUnitPieces[0];
            String expectedUnits = conversionUnitPieces[1];
            if (this.isPerUnit(expectedUnits)) {
                return this.convertEmissionsPerUnit(target, conversion, newUnits, target.getUnits(), false);
            }
            return this.convertEmissionsPerVolume(target, conversion, newUnits, expectedUnits, false);
        }
        if ("%".equals(currentUnits)) {
            BigDecimal originalValue = target.getValue();
            BigDecimal asRatio = originalValue.multiply(TO_PERCENT_FACTOR);
            EngineNumber total = this.stateGetter.getGhgConsumption();
            BigDecimal newValue = total.getValue().multiply(asRatio);
            return new EngineNumber(newValue, "tCO2e");
        }
        throw new IllegalArgumentException("Unable to convert to consumption: " + currentUnits);
    }

    private EngineNumber toKgCo2eConsumption(EngineNumber target) {
        boolean currentInfer;
        String currentUnits = (target = this.normalize(target)).getUnits();
        boolean alreadyCorrect = "kgCO2e".equals(currentUnits) || "kgCO2eeachyear".equals(currentUnits);
        boolean currentVolume = "kg".equals(currentUnits) || "mt".equals(currentUnits);
        boolean currentPop = "unit".equals(currentUnits) || "units".equals(currentUnits);
        boolean bl = currentInfer = currentVolume || currentPop;
        if (alreadyCorrect) {
            return target;
        }
        if (currentInfer) {
            EngineNumber conversion = this.stateGetter.getSubstanceConsumption();
            String normalizedUnits = UnitConverter.normalizeUnitString(conversion.getUnits());
            String[] conversionUnitPieces = UnitConverter.splitUnits(normalizedUnits);
            String newUnits = conversionUnitPieces[0];
            String expectedUnits = conversionUnitPieces[1];
            if (this.isPerUnit(expectedUnits)) {
                return this.convertEmissionsPerUnit(target, conversion, newUnits, target.getUnits(), true);
            }
            return this.convertEmissionsPerVolume(target, conversion, newUnits, expectedUnits, true);
        }
        return switch (currentUnits) {
            case "tCO2e" -> {
                BigDecimal tco2eValue = target.getValue();
                BigDecimal kgco2eValue = tco2eValue.multiply(TCO2E_TO_KGCO2E_FACTOR);
                yield new EngineNumber(kgco2eValue, "kgCO2e");
            }
            case "%" -> {
                BigDecimal originalValue = target.getValue();
                BigDecimal asRatio = originalValue.multiply(TO_PERCENT_FACTOR);
                EngineNumber total = this.stateGetter.getGhgConsumption();
                BigDecimal kgco2eTotal = total.getValue().multiply(TCO2E_TO_KGCO2E_FACTOR);
                BigDecimal newValue = kgco2eTotal.multiply(asRatio);
                yield new EngineNumber(newValue, "kgCO2e");
            }
            default -> throw new IllegalArgumentException("Unable to convert to kgCO2e consumption: " + currentUnits);
        };
    }

    private EngineNumber toEnergyConsumption(EngineNumber target) {
        boolean alreadyCorrect;
        String currentUnits = (target = this.normalize(target)).getUnits();
        boolean currentVolume = "kg".equals(currentUnits) || "mt".equals(currentUnits);
        boolean currentPop = "unit".equals(currentUnits) || "units".equals(currentUnits);
        boolean currentInfer = currentVolume || currentPop;
        boolean bl = alreadyCorrect = "kwh".equals(currentUnits) || "kwheachyear".equals(currentUnits);
        if (alreadyCorrect) {
            return target;
        }
        if (currentInfer) {
            EngineNumber conversion = this.stateGetter.getEnergyIntensity();
            BigDecimal conversionValue = conversion.getValue();
            String normalizedUnits = UnitConverter.normalizeUnitString(conversion.getUnits());
            String[] conversionUnitPieces = UnitConverter.splitUnits(normalizedUnits);
            String newUnits = conversionUnitPieces[0];
            String expectedUnits = conversionUnitPieces[1];
            EngineNumber targetConverted = this.convert(target, expectedUnits);
            BigDecimal originalValue = targetConverted.getValue();
            BigDecimal newValue = originalValue.multiply(conversionValue);
            if (!"kwh".equals(newUnits)) {
                throw new IllegalArgumentException("Unexpected units " + newUnits);
            }
            return new EngineNumber(newValue, newUnits);
        }
        if ("%".equals(currentUnits)) {
            BigDecimal originalValue = target.getValue();
            BigDecimal asRatio = originalValue.multiply(TO_PERCENT_FACTOR);
            EngineNumber total = this.stateGetter.getEnergyConsumption();
            BigDecimal newValue = total.getValue().multiply(asRatio);
            return new EngineNumber(newValue, "kwh");
        }
        throw new IllegalArgumentException("Unable to convert to energy consumption: " + currentUnits);
    }

    private EngineNumber toYears(EngineNumber target) {
        String currentUnits = (target = this.normalize(target)).getUnits();
        if ("years".equals(currentUnits) || this.isYearsUnitStr(currentUnits)) {
            if ("years".equals(currentUnits)) {
                return target;
            }
            return new EngineNumber(target.getValue(), "years");
        }
        return switch (currentUnits) {
            case "tCO2e" -> {
                BigDecimal perYearConsumptionValue = this.stateGetter.getGhgConsumption().getValue();
                BigDecimal newYears = target.getValue().divide(perYearConsumptionValue, MATH_CONTEXT);
                yield new EngineNumber(newYears, "years");
            }
            case "kgCO2e" -> {
                BigDecimal kgco2eValue = target.getValue();
                BigDecimal tco2eValue = kgco2eValue.multiply(KGCO2E_TO_TCO2E_FACTOR);
                BigDecimal perYearConsumptionValue = this.stateGetter.getGhgConsumption().getValue();
                BigDecimal newYears = tco2eValue.divide(perYearConsumptionValue, MATH_CONTEXT);
                yield new EngineNumber(newYears, "years");
            }
            case "kwh" -> {
                BigDecimal perYearConsumptionValue = this.stateGetter.getEnergyConsumption().getValue();
                BigDecimal newYears = target.getValue().divide(perYearConsumptionValue, MATH_CONTEXT);
                yield new EngineNumber(newYears, "years");
            }
            case "kg", "mt" -> {
                EngineNumber perYearVolume = this.stateGetter.getVolume();
                String perYearVolumeUnits = perYearVolume.getUnits();
                BigDecimal perYearVolumeValue = perYearVolume.getValue();
                EngineNumber volumeConverted = this.convert(target, perYearVolumeUnits);
                BigDecimal volumeConvertedValue = volumeConverted.getValue();
                BigDecimal newYears = volumeConvertedValue.divide(perYearVolumeValue, MATH_CONTEXT);
                yield new EngineNumber(newYears, "years");
            }
            case "unit", "units" -> {
                BigDecimal perYearPopulation = this.stateGetter.getPopulationChange(this).getValue();
                BigDecimal newYears = target.getValue().divide(perYearPopulation, MATH_CONTEXT);
                yield new EngineNumber(newYears, "years");
            }
            case "%" -> {
                BigDecimal originalValue = target.getValue();
                BigDecimal asRatio = originalValue.multiply(TO_PERCENT_FACTOR);
                EngineNumber total = this.stateGetter.getYearsElapsed();
                BigDecimal newValue = total.getValue().multiply(asRatio);
                yield new EngineNumber(newValue, "years");
            }
            default -> throw new IllegalArgumentException("Unable to convert to years: " + currentUnits);
        };
    }

    private EngineNumber toPercent(EngineNumber target) {
        String currentUnits = (target = this.normalize(target)).getUnits();
        if ("%".equals(currentUnits)) {
            return target;
        }
        EngineNumber total = switch (currentUnits) {
            case "year", "years", "yr", "yrs" -> this.stateGetter.getYearsElapsed();
            case "tCO2e" -> this.stateGetter.getGhgConsumption();
            case "kgCO2e" -> {
                EngineNumber tco2eTotal = this.stateGetter.getGhgConsumption();
                BigDecimal kgco2eTotal = tco2eTotal.getValue().multiply(TCO2E_TO_KGCO2E_FACTOR);
                yield new EngineNumber(kgco2eTotal, "kgCO2e");
            }
            case "kg", "mt" -> {
                EngineNumber volume = this.stateGetter.getVolume();
                yield this.convert(volume, currentUnits);
            }
            case "unit", "units" -> this.stateGetter.getPopulation();
            default -> throw new IllegalArgumentException("Unable to convert to %: " + currentUnits);
        };
        BigDecimal percentValue = target.getValue().divide(total.getValue(), MATH_CONTEXT).multiply(PERCENT_FACTOR);
        return new EngineNumber(percentValue, "%");
    }

    private EngineNumber normalize(EngineNumber target) {
        target = this.normUnits(target);
        target = this.normTime(target);
        target = this.normConsumption(target);
        target = this.normVolume(target);
        return target;
    }

    private EngineNumber normUnits(EngineNumber target) {
        boolean isPerUnit;
        String currentUnits = target.getUnits();
        String normalizedCurrentUnits = UnitConverter.normalizeUnitString(currentUnits);
        boolean bl = isPerUnit = normalizedCurrentUnits.endsWith("/unit") || normalizedCurrentUnits.endsWith("/units");
        if (!isPerUnit) {
            return target;
        }
        BigDecimal originalValue = target.getValue();
        String newUnits = normalizedCurrentUnits.substring(0, normalizedCurrentUnits.indexOf("/"));
        EngineNumber population = this.stateGetter.getPopulation();
        BigDecimal populationValue = population.getValue();
        BigDecimal newValue = originalValue.multiply(populationValue);
        return new EngineNumber(newValue, newUnits);
    }

    private EngineNumber normTime(EngineNumber target) {
        String currentUnits = target.getUnits();
        String normalizedCurrentUnits = UnitConverter.normalizeUnitString(currentUnits);
        if (!this.getEndsWithPerYear(normalizedCurrentUnits)) {
            return target;
        }
        BigDecimal originalValue = target.getValue();
        String newUnits = normalizedCurrentUnits.substring(0, normalizedCurrentUnits.indexOf("/"));
        EngineNumber years = this.stateGetter.getYearsElapsed();
        BigDecimal yearsValue = years.getValue();
        BigDecimal newValue = originalValue.multiply(yearsValue);
        return new EngineNumber(newValue, newUnits);
    }

    private EngineNumber normConsumption(EngineNumber target) {
        EngineNumber targetConsumption;
        boolean isNormed;
        String currentUnits = target.getUnits();
        String normalizedCurrentUnits = UnitConverter.normalizeUnitString(currentUnits);
        boolean isCo2 = normalizedCurrentUnits.endsWith("/tCO2e");
        boolean isKgCo2 = normalizedCurrentUnits.endsWith("/kgCO2e");
        boolean isKwh = normalizedCurrentUnits.endsWith("/kwh");
        boolean bl = isNormed = !isCo2 && !isKgCo2 && !isKwh;
        if (isNormed) {
            return target;
        }
        if (isCo2) {
            targetConsumption = this.stateGetter.getGhgConsumption();
        } else if (isKgCo2) {
            EngineNumber tco2eConsumption = this.stateGetter.getGhgConsumption();
            BigDecimal kgco2eValue = tco2eConsumption.getValue().multiply(TCO2E_TO_KGCO2E_FACTOR);
            targetConsumption = new EngineNumber(kgco2eValue, "kgCO2e");
        } else {
            targetConsumption = this.stateGetter.getEnergyConsumption();
        }
        BigDecimal originalValue = target.getValue();
        String newUnits = normalizedCurrentUnits.substring(0, normalizedCurrentUnits.indexOf("/"));
        BigDecimal totalConsumptionValue = targetConsumption.getValue();
        BigDecimal newValue = originalValue.multiply(totalConsumptionValue);
        return new EngineNumber(newValue, newUnits);
    }

    private EngineNumber normVolume(EngineNumber target) {
        boolean needsNorm;
        String targetUnits = target.getUnits();
        String normalizedTargetUnits = UnitConverter.normalizeUnitString(targetUnits);
        boolean divKg = normalizedTargetUnits.endsWith("/kg");
        boolean divMt = normalizedTargetUnits.endsWith("/mt");
        boolean bl = needsNorm = divKg || divMt;
        if (!needsNorm) {
            return target;
        }
        String[] targetUnitPieces = UnitConverter.splitUnits(normalizedTargetUnits);
        String newUnits = targetUnitPieces[0];
        String expectedUnits = targetUnitPieces[1];
        EngineNumber volume = this.stateGetter.getVolume();
        EngineNumber volumeConverted = this.convert(volume, expectedUnits);
        BigDecimal conversionValue = volumeConverted.getValue();
        BigDecimal originalValue = target.getValue();
        BigDecimal newValue = originalValue.multiply(conversionValue);
        return new EngineNumber(newValue, newUnits);
    }

    private boolean isPerUnit(String expectedUnits) {
        return "unit".equals(expectedUnits) || "units".equals(expectedUnits);
    }

    private EngineNumber convertEmissionsPerUnit(EngineNumber target, EngineNumber conversion, String newUnits, String targetUnits, boolean isKgCo2eOutput) {
        BigDecimal emissionsValue;
        BigDecimal populationValue;
        if ("unit".equals(targetUnits) || "units".equals(targetUnits)) {
            populationValue = target.getValue();
        } else {
            EngineNumber amortizedVolume = this.stateGetter.getAmortizedUnitVolume();
            BigDecimal volumePerUnit = amortizedVolume.getValue();
            BigDecimal targetVolume = target.getValue();
            populationValue = targetVolume.divide(volumePerUnit, MATH_CONTEXT);
        }
        BigDecimal conversionValue = conversion.getValue();
        if (isKgCo2eOutput) {
            if ("kgCO2e".equals(newUnits)) {
                emissionsValue = populationValue.multiply(conversionValue);
                return new EngineNumber(emissionsValue, "kgCO2e");
            }
            if ("tCO2e".equals(newUnits)) {
                BigDecimal tco2eValue = populationValue.multiply(conversionValue);
                BigDecimal kgco2eValue = tco2eValue.multiply(TCO2E_TO_KGCO2E_FACTOR);
                return new EngineNumber(kgco2eValue, "kgCO2e");
            }
            throw new IllegalArgumentException("Unsupported per-unit emissions type: " + newUnits);
        }
        if ("tCO2e".equals(newUnits)) {
            emissionsValue = populationValue.multiply(conversionValue);
            return new EngineNumber(emissionsValue, "tCO2e");
        }
        if ("kgCO2e".equals(newUnits)) {
            BigDecimal kgco2eValue = populationValue.multiply(conversionValue);
            BigDecimal tco2eValue = kgco2eValue.multiply(KGCO2E_TO_TCO2E_FACTOR);
            return new EngineNumber(tco2eValue, "tCO2e");
        }
        throw new IllegalArgumentException("Unsupported per-unit emissions type: " + newUnits);
    }

    private EngineNumber convertEmissionsPerVolume(EngineNumber target, EngineNumber conversion, String newUnits, String expectedUnits, boolean isKgCo2eOutput) {
        EngineNumber targetConverted = this.convert(target, expectedUnits);
        BigDecimal originalValue = targetConverted.getValue();
        BigDecimal conversionValue = conversion.getValue();
        BigDecimal newValue = originalValue.multiply(conversionValue);
        if (!"tCO2e".equals(newUnits) && !"kgCO2e".equals(newUnits)) {
            throw new IllegalArgumentException("Unexpected units " + newUnits);
        }
        if (isKgCo2eOutput) {
            if ("tCO2e".equals(newUnits)) {
                newValue = newValue.multiply(TCO2E_TO_KGCO2E_FACTOR);
                newUnits = "kgCO2e";
            }
        } else if ("kgCO2e".equals(newUnits)) {
            newValue = newValue.multiply(KGCO2E_TO_TCO2E_FACTOR);
            newUnits = "tCO2e";
        }
        return new EngineNumber(newValue, newUnits);
    }
}

